<?php
/**
 * GlobalSign OneclickSSL
 *
 * Replacing the slow and error prone process of CSR creation, key management,
 * approver emails and Certificate installation with a single click!
 *
 * PHP version 5
 *
 * LICENSE: BSD License
 *
 * Copyright © 2012 GMO GlobalsSign KK.
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY GMO GLOBALSIGN KK "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * @copyright  Copyright © 2012 GMO GlobalsSign KK. All Rights Reserved. (http://www.globalsign.com)
 * @license    BSD License (3 Clause)
 * @version    $Id$
 * @link       http://www.globalsign.com/ssl/oneclickssl/
 * @link       http://globalsign.github.com/OneClickSSL/
 */

require 'OneClickSSLPlugin.php';
require 'CertificateData.php';
require 'OneClickService.php';
require 'Output/Output.php';
require 'Output/Debug.php';
require 'Output/Status.php';

/**
 * --------------------------------------------------------
 *        !!! PLEASE DO NOT EDIT THIS FILE !!!
 *      Use extentions to modify the functionality
 *
 *      Provide a patch for a bugfix or new feature
 *       https://github.com/GlobalSign/OneClickSSL
 * --------------------------------------------------------
 */

/**
 * OneClickSSL
 */
class OneClickSSL
{
    const PROD_ENV_ID  = 0;

    const TEST_ENV_ID  = 1;

    const STGE_ENV_ID  = 2;

    const DEFAULT_ENV  = self::PROD_ENV_ID;

    const STATUS_BUSY  = 'busy';

    const STATUS_ERROR = 'error';

    const STATUS_DONE  = 'done';

    /**
     * The Plugin to integrate with
     *
     * @var OneClickSSLPlugin
     */
    protected $_plugin;

    /**
     * Output container for debug and status writing
     *
     * @var Output_Output
     */
    protected $_output;

    /**
     * Certificate data container
     *
     * @var CertificateData
     */
    protected $_certData;

    /**
     * OneClicKSSL Service class
     *
     * @var OneClickService
     */
    protected $_service;

    /**
     * Flag that we've made a backup
     *
     * @var bool
     */
    private $_hasBackup = false;

    /**
     * Initialise the object and set-up some default loggerers
     * Short-cut so third-parties don't have to worry about setting up debug logger and status
     *
     * @param CertificateData   $certData  Certificate Data container
     * @param OneClickSSLPlugin $plugin    Plugin object to use
     *
     * @return null
     */
    public static function init(CertificateData $certData, OneClickSSLPlugin $plugin)
    {
        return new self($certData, $plugin, new Output_Output(new Output_Debug(), new Output_Status()));
    }

    /**
     * Set up the object with the Plugin and Output handler
     *
     * @param CertificateData   $certData  Certificate Data container
     * @param OneClickSSLPlugin $plugin    Plugin object to use
     * @param Output_Output     $output    Output handler to use
     */
    public function __construct(CertificateData $certData, OneClickSSLPlugin $plugin, Output_Output $output)
    {
        $this->_certData = $certData;
        $this->_plugin   = $plugin;
        $this->_output   = $output;

        $this->_plugin->setOutput($output);
        $this->_plugin->setDomain($certData->getDomain());
    }

    /**
     * Return the output handler
     *
     * @return Output_Output
     */
    public function output()
    {
        return $this->_output;
    }

    /**
     * Set the value for the particular environment we are using
     *
     * @param int $value  Environment ID
     *
     * @return OneClickSSL
     */
    public function setEnvironment($value)
    {
        if (!in_array($value, array(self::PROD_ENV_ID, self::TEST_ENV_ID, self::STGE_ENV_ID))) {
            throw new InvalidArgumentException('Invalid environment ID');
        }

        $this->_service = $this->getOneClickService($value);

        return $this;
    }

    /**
     * Access the OneClickService wrapper
     *
     * @return OneClickService
     */
    public function service()
    {
        if (!isset($this->_service)) {
            $this->_service = $this->getOneClickService(self::DEFAULT_ENV);
        }

        return $this->_service;
    }

    /**
     * Order a certificate
     *
     * Now we actually use all details and order the certificate
     *
     * @return int  Status value of operation. 0 is success
     */
    public function order()
    {
        $restoreBackupBag = array('restoreBackup' => 'restoreBackup');

        $workflow = array(
            'checkIp'       => array(
                'func'   => 'checkIp',
                'blob'   => 'Checking the IP Address'),
            'checkVoucher'  => array(
                'func'   => 'checkVoucherForOrder',
                'blob'   => 'Verification of domain and voucher details'),
            'doBackup'      => array(
                'func'   => 'backup',
                'blob'   => 'Backing up the current certificate'),
            'newPrivateKey' => array(
                'func'   => 'newPrivateKey',
                'onFail' => $restoreBackupBag,
                'blob'   => 'Generating new private key'),
            'testOrder'     => array(
                'func'   => 'testOrder',
                'onFail' => $restoreBackupBag,
                'blob'   => 'Placing the temporary order'),
            'installTest'   => array(
                'func'   => 'installTest',
                'onFail' => $restoreBackupBag,
                'blob'   => 'Installing the test certificate'),
            'realOrder'     => array(
                'func'   => 'realOrder',
                'onFail' => $restoreBackupBag,
                'blob'   => 'Placing the real order'),
            'installReal'   => array(
                'func'   => 'installReal',
                'onFail' => $restoreBackupBag,
                'blob'   => 'Installing the real certificate'),
        );

        // Not very pretty but we want to remove these items from the flow if these
        // methods aren't implemented in the plug-in
        if (!method_exists($this->_plugin, 'checkIp')) {
            unset($workflow['checkIp']);
        }
        if (!method_exists($this->_plugin, 'backup')) {
            unset($workflow['doBackup']);
        }

        $result = $this->processWorkflow($workflow);
        if ($result) {
            $this->status('orderDone', self::STATUS_DONE);
            $this->updateStatus();
        }

        return $result;
    }

    /**
     * Revoke a certificate
     *
     * @return int  Status value of operation. 0 is success
     */
    public function revoke()
    {
        $restoreBackupBag = array('restoreBackup' => 'restoreBackup');

        $workflow = array(
            'checkVoucher'  => array(
                'func'   => 'checkVoucherForRevoke',
                'blob'   => 'Verification of domain and voucher details'),
            'doBackup'      => array(
                'func'   => 'backup',
                'blob'   => 'Backing up the current certificate'),
            'newPrivateKey' => array(
                'func'   => 'newPrivateKey',
                'onFail' => $restoreBackupBag,
                'blob'   => 'Generating new private key'),
            'testOrder'     => array(
                'func'   => 'testOrder',
                'onFail' => $restoreBackupBag,
                'blob'   => 'Placing the temporary order'),
            'installTest'   => array(
                'func'   => 'installTest',
                'onFail' => $restoreBackupBag,
                'blob'   => 'Installing the test certificate'),
            'revokeCert'    => array(
                'func'   => 'revokeCert',
                'onFail' => $restoreBackupBag,
                'blob'   => 'Revoking the certificate'),
            'restoreBackup' => array(
                'func'   => 'restoreBackup',
                'blob'   => 'Restoring the old certificate'),
        );

        // Not very pretty but we want to remove these items from the flow if these
        // methods aren't implemented in the plug-in.
        if (!method_exists($this->_plugin, 'backup')) {
            unset($workflow['doBackup']);
        }
        if (!method_exists($this->_plugin, 'restoreBackup')) {
            unset($workflow['restoreBackup']);
        }

        $result = $this->processWorkflow($workflow);
        if ($result) {
            $this->status('revokeDone', self::STATUS_DONE);
            $this->updateStatus();
        }

        return $result;
    }

    /**
     * Check the IP
     *
     * @return bool
     */
    protected function checkIp()
    {
        // Check for unique ip address (if this is not a revocation
        if (method_exists($this->_plugin, 'checkIp')) {
            // Cancel request if we have no unique ip (and there could be not assigned one)
            if (!$this->_plugin->checkIp()) {
                return false;
            }
        }
        return true;
    }

    /**
     * checkVoucherForOrder
     *
     * @return bool
     */
    protected function checkVoucherForOrder()
    {
        return $this->checkVoucher(false);
    }

    /**
     * checkVoucherForRevoke
     *
     * @return bool
     */
    protected function checkVoucherForRevoke()
    {
        return $this->checkVoucher(true);
    }

    /**
     * Check for a valid voucher
     *
     * @param mixed $revoke
     * @return null
     */
    protected function checkVoucher($revoke = false)
    {
        if (!$this->service()->checkVoucher($revoke)) {
            throw new RunTimeException(
                'Failed to validate voucher code',
                $this->service()->getCheckVoucherResponseCode()
            );
        }

        return true;
    }

    /**
     * backup
     *
     * @return null
     */
    protected function backup()
    {
        // Do backup the current certificates
        if (method_exists($this->_plugin, 'backup')) {
            if ($this->_plugin->backup()) {
                $this->_hasBackup = true;
            }
        }

        return true;
    }

    /**
     * generatePrivateKey
     *
     * @return null
     */
    protected function newPrivateKey()
    {
        // FIXME: Following code feels a bit wrong. This should all be contained inside the Service class?
        $this->service()->newPrivateKey($this->service()->getKeyLength());
        $csr = $this->service()->newCsr($this->service()->getPrivateKey());
        openssl_csr_export($csr, $csrTxt);
        $this->service()->setCsrTxt($csrTxt);

        return true;
    }

    /**
     * testOrder
     *
     * @return null
     */
    protected function testOrder()
    {
        // Place the test (temporary) order
        if (!$this->service()->testOrder($this->service()->getVoucherOrderId(), $this->service()->getCsrTxt())) {
            throw new RunTimeException(
                'Order for the test (temporary) certifiacte failed',
                $this->service()->getTestResponseCode()
            );
        }

        return true;
    }

    /**
     * installTest
     *
     * @return null
     */
    protected function installTest()
    {
        $certHash = $this->_plugin->install(
            $this->service()->getPrivateKey(),
            $this->service()->getTestCertificate(),
            $this->service()->getTestCaCert()
        );
        return ($certHash) ? $this->service()->checkInstall($certHash) : false;
    }

    /**
     * revokeCert
     *
     * @return null
     */
    protected function revokeCert()
    {
        // When a certificate is revoked no new production certificate is installed
        // - The "Voucher" is the appropriate serial number of the SSL certificate to be revoked.
        if (!$this->service()->revokeCert($this->service()->getTestOrderId())) {
            throw new RunTimeException(
                'Failed to revoke the certificate',
                $this->service()->getRevokeResponseCode()
            );
        }

        // Revocation completed
        return true;
    }

    /**
     * realOrder
     *
     * @return null
     */
    protected function realOrder()
    {
        if (!$this->service()->realOrder($this->service()->getTestOrderId(), $this->service()->getCsrTxt())) {
            throw new RunTimeException(
                'Order for the real certifiacte failed',
                $this->service()->getOrderResponseCode()
            );
        }

        return true;
    }

    /**
     * installReal
     *
     * @return null
     */
    protected function installReal()
    {
        // Install the real certificate
        $certHash = $this->_plugin->install(
            $this->service()->getPrivateKey(),
            $this->service()->getRealCertificate(),
            $this->service()->getRealCaCert()
        );
        return ($certHash) ? $this->service()->checkInstall($certHash) : false;
    }

    /**
     * Restore the backup if it exists
     *
     * @return bool
     */
    protected function restoreBackup()
    {
        if ($this->_hasBackup && method_exists($this->_plugin, 'restoreBackup')) {
            return $this->_plugin->restoreBackup();
        }
        return true;
    }

    /**
     * getOneClickService
     *
     * @param mixed $environment
     * @return null
     */
    protected function getOneClickService($environment)
    {
        // Wrapped this instantiation so we can override it when testing. Some real
        // dependency injection container would be better.
        return new OneClickService($this->_certData, $environment, $this->_output);
    }

    /**
     * Process a chain of commands - poor man's state machine
     * Ideally this will be moved to another class and each workflow item will be
     * a separate implementation of a workitem interface.
     *
     * @param array $workflow  An array of work items
     *
     * @return bool
     */
    protected function processWorkflow($workflow)
    {
        $this->debug(2, 'Processing workflow (' . implode(', ', array_keys($workflow)) . ')');
        $start = microtime('true');

        foreach ($workflow as $handle => $bag) {
            $funcStart = microtime('true');

            if (!$this->executeWorkitem($handle, $bag['func'])) {
                if (isset($bag['onFail'])) {
                    $failHandle = key($bag['onFail']);
                    $this->executeWorkitem($failHandle, $bag['onFail'][$failHandle]);
                }
                return false;
            }

            $this->debug(1, "{$bag['blob']} took " . number_format(microtime(true) - $funcStart, 2) . ' seconds');
        }

        $this->debug(1, 'The whole process completed in ' . number_format(microtime(true) - $start, 2) . ' seconds');

        return true;
    }

    /**
     * Execute an individual work item.
     *
     * @param mixed $func
     *
     * @return null
     */
    protected function executeWorkitem($handle, $func)
    {
        $this->status($handle, self::STATUS_BUSY);
        $this->updateStatus();

        try {
            if (!method_exists($this, $func)) {
                $this->status($handle, $error);
                $this->debug(1, "Work item '{$func}' not found");
                return false;
            }
            $result = call_user_func(array($this, $func));
            if ($result === false) {
                $this->status($handle, self::STATUS_ERROR);
                $this->updateStatus();
                return $result;
            }
        } catch (RunTimeException $e) {
            $this->debug(1, $e->getMessage());
            $this->status($handle, self::STATUS_ERROR);
            $this->updateStatus();
            return false;
        }

        $this->status($handle, self::STATUS_DONE);

        return true;
    }

    /**
     * debug
     *
     * @param int    $level    Level to display message for
     * @param string $message  Message to display
     *
     * @return null
     */
    protected function debug($level, $message)
    {
        $this->output()->debug()->write($level, $message);
    }

    /**
     * status
     *
     * @param mixed $key
     * @param mixed $value
     * @return null
     */
    protected function status($key, $value)
    {
        $this->output()->status()->write($key, $value);
    }

    /**
     * Update the status file, if enabled
     *
     * @return null
     */
    protected function updateStatus()
    {
        try {
            if (!$this->output()->status()->updateStatus($this->_certData->getDomain())) {
                $this->debug(3, 'Skip writing status information to file');
            }
        } catch (RunTimeException $e) {
            $this->debug(1, $e->getMessage());
        }
    }
}
